<?php

namespace SEOPressPro\Services\OpenAI;

defined('ABSPATH') || exit;

class Completions {
    private const OPENAI_URL_CHAT_COMPLETIONS = 'https://api.openai.com/v1/chat/completions';

    /**
     * Generate titles and descriptions for a post.
     *
     * This function generates titles and descriptions based on the provided parameters.
     *
     * @param int    $post_id   The ID of the post for which to generate titles and descriptions.
     * @param string $meta      title|desc (optional).
     * @param string $language  The language for generating titles and descriptions (default is 'en_US').
     * @param bool   $autosave  Whether this is an autosave operation, useful for bulk actions (default is false).
     *
     * @return array $data The answers from OpenAI with title/desc
     */
    public function generateTitlesDesc($post_id, $meta = '', $language = 'en_US', $autosave = false) {
        //Init
        $title = '';
        $description = '';
        $message = '';

        $content = get_post_field('post_content', $post_id);
        $content = esc_attr(stripslashes_deep(wp_filter_nohtml_kses(wp_strip_all_tags(strip_shortcodes($content)))));

        //Compatibility with current page and theme builders
        $theme = wp_get_theme();

        //Divi
        if ('Divi' == $theme->template || 'Divi' == $theme->parent_theme) {
            $regex = '/\[(\[?)(et_pb_[^\s\]]+)(?:(\s)[^\]]+)?\]?(?:(.+?)\[\/\2\])?|\[\/(et_pb_[^\s\]]+)?\]/';
            $content = preg_replace($regex, '', $content);
        }

        //Bricks compatibility
        if (defined('BRICKS_DB_EDITOR_MODE') && ('bricks' == $theme->template || 'Bricks' == $theme->parent_theme)) {
            $page_sections = get_post_meta($post_id, BRICKS_DB_PAGE_CONTENT, true);
            $editor_mode   = get_post_meta($post_id, BRICKS_DB_EDITOR_MODE, true);

            if (is_array($page_sections) && 'wordpress' !== $editor_mode) {
                $content = \Bricks\Frontend::render_data($page_sections);
            }
        }

        //Limit post content sent to 500 words (higher value will return a 400 error)
        $content = wp_trim_words( $content, 500 );

        //If no post_content use the permalink
        if (empty($content)) {
            $content = get_permalink($post_id);
        }

        $api_key = seopress_pro_get_service('Usage')->getLicenseKey();

        //Get OpenAI model
        $options = get_option('seopress_pro_option_name');
        $model = isset($options['seopress_ai_openai_model']) ? $options['seopress_ai_openai_model'] : 'gpt-3.5-turbo';

        if ($api_key ==='xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx' || empty($api_key)) {
            $message = __('Your OpenAI API key has not been entered correctly. Please paste it again from OpenAI website to the settings.', 'wp-seopress-pro');

            $data = [
                'message' => $message,
            ];

            return $data;
        }

        $body = [
            'model'    => $model,
            'temperature' => 1,
            'max_tokens' => 220,
        ];

        $body['messages'] = [];

        //Get current post language for bulk actions
        if ($meta === 'title' || $meta === 'desc') {
            //Default
            $language = get_locale();

            //WPML
            if (defined('ICL_SITEPRESS_VERSION')) {
                $language = apply_filters( 'wpml_post_language_details', NULL, $post_id );
                $language = !empty($language['locale']) ? $language['locale'] : get_locale();
            }

            //Polylang
            if (function_exists('pll_get_post_language')) {
                $language = !empty(pll_get_post_language($post_id, 'locale')) ? pll_get_post_language($post_id, 'locale') : get_locale();
            }
        }

        //Convert language code to readable name
        $language = locale_get_display_name($language, 'en') ? esc_html(locale_get_display_name($language, 'en')) : $language;

        //Prompt for meta title
        $prompt_title = sprintf(__('Generate, in this language %1$s, an engaging SEO title metadata in one sentence of sixty characters maximum for this article: %2$s', 'wp-seopress-pro'), esc_attr($language), esc_html($content));
        $msg   = apply_filters( 'seopress_ai_openai_meta_title', $prompt_title, $post_id );

        if (empty($meta) || $meta === 'title') {
            $body['messages'][] = ['role' => 'user', 'content' => $msg];
        }

        //Prompt for meta description
        $prompt_desc = sprintf(__('Generate, in this language ' . $language . ', an engaging SEO meta description in less than 160 characters for this article: %2$s', 'wp-seopress-pro'), esc_attr($language), esc_html($content));
        $msg   = apply_filters( 'seopress_ai_openai_meta_desc', $prompt_desc, $post_id );

        if (empty($meta) || $meta === 'desc') {
            $body['messages'][] = ['role' => 'user', 'content' => $msg];
        }

        $body['messages'][] = ['role' => 'user', 'content' => 'Provide the answer as an object with "title" as first key and "desc" for second key for parsing in this language ' . $language . '.'];

        $args = [
            'body'        => json_encode($body),
            'timeout'     => '30',
            'redirection' => '5',
            'httpversion' => '1.0',
            'blocking'    => true,
            'headers' => [
                'Authorization' => 'Bearer ' . $api_key,
                'Content-Type' => 'application/json'
            ],
        ];

        $args = apply_filters('seopress_ai_openai_request_args', $args);

        $url = self::OPENAI_URL_CHAT_COMPLETIONS;

        $response = wp_remote_post( $url, $args );

        // make sure the response came back okay
        if (is_wp_error($response) || 200 !== wp_remote_retrieve_response_code($response)) {
            if (is_wp_error($response)) {
                $message = $response->get_error_message();
            } else {
                $message = __('An error occurred, please try again. Response code: ', 'wp-seopress-pro') . wp_remote_retrieve_response_code($response);
            }
        } else {
            $data = json_decode(wp_remote_retrieve_body($response), true);

            $message = 'Success';

            if (empty($meta) || $meta === 'title') {
                $result = json_decode($data['choices'][0]['message']['content'], true);
                $result = $result['title'];

                $title = esc_attr(trim(stripslashes_deep(wp_filter_nohtml_kses(wp_strip_all_tags(strip_shortcodes($result)))), '"'));

                if ($autosave === true) {
                    update_post_meta( $post_id, '_seopress_titles_title', $title);
                }
            }

            if (empty($meta)) {
                $result = json_decode($data['choices'][0]['message']['content'], true);
                $result = $result['desc'];
            } elseif ($meta === 'desc') {
                $result = json_decode($data['choices'][0]['message']['content'], true);
                $result = $result['desc'];
            }

            if (empty($meta) || $meta === 'desc') {
                $description = esc_attr(trim(stripslashes_deep(wp_filter_nohtml_kses(wp_strip_all_tags(strip_shortcodes($result)))), '"'));

                if ($autosave === true) {
                    update_post_meta( $post_id, '_seopress_titles_desc', $description);
                }
            }
        }

        $data = [
            'message' => $message,
            'title' => $title,
            'desc' => $description
        ];

        return $data;
    }
}
