<?php

namespace Sendcloud\Shipping\Checkout\Storage;

use SendCloud\Checkout\Contracts\Storage\CheckoutStorage;
use SendCloud\Checkout\Domain\Delivery\DeliveryMethod;
use SendCloud\Checkout\Domain\Delivery\DeliveryZone;
use Sendcloud\Shipping\Repositories\Delivery_Methods_Repository;
use Sendcloud\Shipping\Repositories\Delivery_Zones_Repository;

/**
 * Class Checkout_Storage
 *
 * @package Sendcloud\Shipping\Checkout\Storage
 */
class Checkout_Storage implements CheckoutStorage {
	/**
	 * Delivery_Zones_Repository
	 *
	 * @var Delivery_Zones_Repository
	 */
	private $delivery_zones_repository;
	/**
	 * Delivery_Methods_Repository
	 *
	 * @var Delivery_Methods_Repository
	 */
	private $delivery_methods_repository;

	/**
	 * Checkout_Storage constructor.
	 *
	 * @param Delivery_Zones_Repository $delivery_zones_repository
	 * @param Delivery_Methods_Repository $delivery_methods_repository
	 */
	public function __construct( $delivery_zones_repository, $delivery_methods_repository ) {
		$this->delivery_zones_repository   = $delivery_zones_repository;
		$this->delivery_methods_repository = $delivery_methods_repository;
	}

	/**
	 * Provides all delivery zone configurations.
	 *
	 * @return DeliveryZone[]
	 */
	public function findAllZoneConfigs() {
		return $this->delivery_zones_repository->find_all();
	}

	/**
	 * Provides delivery zones with specified ids.
	 *
	 * @param array $ids
	 *
	 * @return DeliveryZone[]
	 */
	public function findZoneConfigs( array $ids ) {
		return $this->delivery_zones_repository->find( $ids );
	}

	/**
	 * Deletes specified zone configurations.
	 *
	 * @param array $ids
	 *
	 * @return void
	 */
	public function deleteSpecificZoneConfigs( array $ids ) {
		$this->delivery_zones_repository->delete( $ids );
	}

	/**
	 * Deletes all saved zone configurations.
	 *
	 * @return void
	 */
	public function deleteAllZoneConfigs() {
		$this->delivery_zones_repository->delete_all();
	}

	/**
	 * Creates delivery zones.
	 *
	 * @param DeliveryZone[] $zones
	 *
	 * @return void
	 */
	public function createZoneConfigs( array $zones ) {
		$this->delivery_zones_repository->create( $zones );
	}

	/**
	 * Updates saved zone configurations.
	 *
	 * @param DeliveryZone[] $zones
	 *
	 * @return void
	 */
	public function updateZoneConfigs( array $zones ) {
		$this->delivery_zones_repository->update( $zones );
	}

	/**
	 * Provides all delivery method configurations.
	 *
	 * @return DeliveryMethod[]
	 */
	public function findAllMethodConfigs() {
		return $this->delivery_methods_repository->find_all();
	}

	/**
	 * Deletes methods identified by the provided batch of ids.
	 *
	 * @param array $ids
	 *
	 * @return void
	 */
	public function deleteSpecificMethodConfigs( array $ids ) {
		$this->delivery_methods_repository->delete( $ids );
	}

	/**
	 * Deletes all delivery method configurations.
	 *
	 * @return void
	 */
	public function deleteAllMethodConfigs() {
		$this->delivery_methods_repository->delete_all();
	}

	/**
	 * Updates saved delivery methods.
	 *
	 * @param DeliveryMethod[] $methods
	 *
	 * @return void
	 */
	public function updateMethodConfigs( array $methods ) {
		$this->delivery_methods_repository->update( $methods );
	}

	/**
	 * Creates method configurations.
	 *
	 * @param DeliveryMethod[] $methods
	 *
	 * @return void
	 */
	public function createMethodConfigs( array $methods ) {
		$this->delivery_methods_repository->create( $methods );
	}

	/**
	 * Deletes all delivery method data generated by the integration.
	 *
	 * @return void
	 */
	public function deleteAllMethodData() {
		// Intentionally left empty.
	}

	/**
	 * Finds delivery methods for specified zone ids.
	 *
	 * @param array $zoneIds
	 *
	 * @return DeliveryMethod[]
	 */
	public function findMethodInZones( array $zoneIds ) {
		return $this->delivery_methods_repository->find_in_zones( $zoneIds );
	}

	/**
	 * Delete delivery method configs for delivery methods that no longer exist in system.
	 *
	 * @return void
	 */
	public function deleteObsoleteMethodConfigs() {
		$this->delivery_methods_repository->delete_obsolete_method_configs();
	}

	/**
	 * Delete delivery zone configs for delivery zones that no longer exist in system.
	 *
	 * @return void
	 */
	public function deleteObsoleteZoneConfigs() {
		$this->delivery_zones_repository->delete_obsolete_zone_configs();
	}
}
