<?php

namespace Sendcloud\Shipping\Checkout\Services;

use SendCloud\Checkout\API\Checkout\Delivery\Zone\DeliveryZone;
use SendCloud\Checkout\Domain\Delivery\DeliveryMethod;
use Sendcloud\Shipping\Repositories\Delivery_Methods_Repository;
use Sendcloud\Shipping\Repositories\Delivery_Zones_Repository;

class Zones_Display_Service {

	const DELIVERY_METHOD_EDIT_URL_PATTERN = 'https://app.sendcloud.com/v2/checkout/configurations/{integration_id}/delivery-zones/{zone_id}/delivery-methods/{method_id}?type={type}';
	/**
	 * Delivery Zones Repository
	 *
	 * @var Delivery_Zones_Repository
	 */
	private $zones_repository;
	private $methods_repository;

	public function __construct() {
		global $wpdb;
		$this->zones_repository   = new Delivery_Zones_Repository($wpdb);
		$this->methods_repository = new Delivery_Methods_Repository($wpdb);
	}

	public function get_zones_with_methods() {
		$shipping_zones = $this->zones_repository->find_all();
		$zone_ids = array_map(function ( $shipping_zone) {
			return $shipping_zone->getId();
		}, $shipping_zones);

		$delivery_methods_dtos = $this->get_delivery_methods($zone_ids);

		/**
		 * Delivery zone DTOs
		 *
		 * @var DeliveryZone[] $zone_dtos
		 */
		$zone_dtos = $this->transform_to_dto($shipping_zones);

		foreach ($zone_dtos as $zone_dto) {
			if (array_key_exists($zone_dto->getId(), $delivery_methods_dtos)) {
				$zone_dto->setDeliveryMethods($delivery_methods_dtos[$zone_dto->getId()]);
			}
		}

		return $zone_dtos;
	}

	/**
	 * Transform shipping zones to DeliveryZone DTOs
	 *
	 * @param \SendCloud\Checkout\Domain\Delivery\DeliveryZone[] $shipping_zones
	 */
	private function transform_to_dto( array $shipping_zones) {
		$raw_data = array_map(function ( $shipping_zone) {
			return json_decode($shipping_zone->getRawConfig(), true);
		}, $shipping_zones);

		return DeliveryZone::fromArrayBatch($raw_data);
	}

	/**
	 * Get delivery methods
	 *
	 * @param $zone_ids
	 *
	 * @return array
	 */
	private function get_delivery_methods( $zone_ids) {
		/**
		 * Delivery methods
		 *
		 * @var DeliveryMethod[] $delivery_methods
		 */
		$delivery_methods = $this->methods_repository->find_in_zones($zone_ids);
		$dtos = array();
		foreach ($delivery_methods as $delivery_method) {
			$dtos[$delivery_method->getDeliveryZoneId()][] = \SendCloud\Checkout\API\Checkout\Delivery\Method\DeliveryMethod::fromArray(json_decode($delivery_method->getRawConfig(), true));
		}

		return $dtos;
	}
}
