<?php

namespace Woocommerce_Preorders;

class Checkout {
	/**
	 * @var mixed
	 */
	private $preordersMode;
	/**
	 * @var mixed
	 */
	private $cart;
	/**
	 * @var mixed
	 */
	private $emailIds;

	public function __construct() {
		$this->preordersMode = get_option( 'wc_preorders_mode' );

		$this->cart = new Cart();
		if ( 'either' === $this->preordersMode ) {
			add_filter( 'woocommerce_add_to_cart_validation', [$this->cart, 'allowOneTypeOnly'], 99, 2 );
		}

		add_action( 'woocommerce_checkout_update_order_meta', [$this, 'managePreOrders'], 10, 2 );
		add_action( 'woocommerce_order_status_changed', [$this, 'emailNotifications'], 10, 4 );
		add_filter( 'woocommerce_payment_complete_order_status', [$this, 'setPreroderStatus'], 10, 3 );
		add_filter( 'woocommerce_billing_fields', [$this, 'addShippingDateField'] );
	}
	/**
	 * Set main order status 'pre-ordered' after payment complete
	 *
	 * @param  [string] $status
	 * @param  [int]    $order_id
	 * @param  [type]   $order
	 * @return status
	 */
	public function setPreroderStatus( $status, $order_id, $order ) {
		/*if ( get_post_meta( $order_id, '_preorder_date', true ) ) {
			return 'pre-ordered';
		}*/
		$order = wc_get_order( $order_id );
		if ( $order->get_meta('_preorder_date') ) {
			return 'pre-ordered';
		}
		return $status;
	}
	/**
	 * send preorder related emails
	 *
	 * @param  [int]    $order_id
	 * @param  [string] $old_status
	 * @param  [string] $new_status
	 * @param  [object] $order
	 * @return void
	 */
	public function emailNotifications( $order_id, $old_status, $new_status, $order ) {
		$valid_old_statuses = ( $old_status == 'pending' || $old_status == 'on-hold' || $old_status == 'failed' );
		if ( $valid_old_statuses && is_checkout() && $new_status == 'pre-ordered' ) {

			// Send "New Email" notification (to customer)
			WC()->mailer()->get_emails()['WC_New_Customer_Pre_Order_Email']->trigger( $order_id );
			// Send "New Email" notification (to admin)
			WC()->mailer()->get_emails()['WC_New_Pre_Order_Email']->trigger( $order_id );
		}
	}

	/**
	 * Sends normal order and invoice email to the customer when the user arrives to the thank you page.
	 */
	public function sendOrderEmail( $orderId ) {
		$orderObj             = wc_get_order( $orderId );
		$email_new_order      = WC()->mailer()->get_emails()['WC_Email_New_Order'];
		$emailProcessingOrder = WC()->mailer()->get_emails()['WC_Email_Customer_Processing_Order'];
		$emailOnHoldOrder     = WC()->mailer()->get_emails()['WC_Email_Customer_On_Hold_Order'];
		$emailCompletedOrder  = WC()->mailer()->get_emails()['WC_Email_Customer_Completed_Order'];

		$hasPreorderedProductsOnly = count( $orderObj->get_items() ) === count( $this->getPreorderedProducts( $orderObj ) );

		// We're only firing these emails if there's only a non-preordered product present.
		if ( !$hasPreorderedProductsOnly ) {
			$email_new_order->trigger( $orderId );
			if ( $orderObj->get_status() == 'on-hold' ) {
				$emailOnHoldOrder->trigger( $orderId );
			} elseif ( $orderObj->get_status() == 'processing' ) {
				$emailProcessingOrder->trigger( $orderId );
			} elseif ( $orderObj->get_status() == 'completed' ) {
				$emailCompletedOrder->trigger( $orderId );
			}
		}
	}

	/**
	 * @param  $fields
	 * @return mixed
	 */
	public function addShippingDateField( $fields ) {
		if ( !is_checkout() && !is_cart() ) {
			return $fields;
		}
		if ( 'no' === get_option( 'wc_preorders_always_choose_date' ) ) {
			$class = ['disabled-input', 'form-row-wide'];
		} else {
			$class = ['form-row-wide'];
		}
		global $woocommerce;
		$cart = $woocommerce->cart->get_cart();
		$this->cart->checkPreOrderProducts( $cart );
		if ( \count( $this->cart->getPreOrderProducts() ) > 0 ) {

			$oldestDate = str_replace( [' 00:00:00'], [''], $this->cart->getOldestDate() );

			$fields['preorder_date'] = [
				'label'             => __( 'Pre order Date', 'pre-orders-for-woocommerce' ),
				'type'              => 'text',
				'class'             => $class,
				'description'       => __( 'Please enter the date when you want to receive your order', 'pre-orders-for-woocommerce' ),
				// 'input_class'   => 'datepicker',
				'priority'          => 35,
				'required'          => true,
				'default'           => $oldestDate,
				'custom_attributes' => ['data-pre_order_date' => $oldestDate],
			];
		}

		return $fields;
	}

	/**
	 * @param  $rates
	 * @param  $package
	 * @return mixed
	 */
	public function manageShippingCosts( $rates, $package ) {
		$factor = 1;
		if ( 'individual' === $this->preordersMode ) {
			/**
			 * If we are on "individual" mode, then we will have to multiply it by the number of
			 * orders that we are going to generate.
			 */

			global $woocommerce;
			$cart = $woocommerce->cart->get_cart();
			$this->cart->checkPreOrderProducts( $cart );
			if ( \count( $this->cart->getPreOrderProducts() ) > 0 ) {
				$factor = 1+\count( $this->cart->getPreOrderProducts() );
			}
		} elseif ( 'partial' === $this->preordersMode ) {
			/*
				* If we are in partial mode and the "multiply shipping" option is enabled,
				* then we will have to multiply our shipping costs by 2
			*/
			$factor = 2;
		}
		foreach ( $rates as $id => $rate ) {
			$rates[$id]->cost *= $factor;
		}

		return $rates;
	}

	/**
	 * @param $orderId
	 * @param $data
	 */
	public function managePreOrders( $orderId, $data ) {
		$order = wc_get_order( $orderId );

		// Calculate shipping split factor
		$factor = $this->getSplitShippingFactor( $order );

		/**
		 * Case #1: treat the whole order as a pre-order
		 * Check if the order is of type partial or individual, and if not set the whole order as pre-ordered
		 */
		if ( isset( $data['preorder_date'] ) ) {
			$order->update_meta_data( '_preorder_date', esc_attr( $data['preorder_date'] ) );
			$order->save();
			//update_post_meta( $orderId, '_preorder_date', esc_attr( $data['preorder_date'] ) );
		}

		/**
		 * Option number 4 is treated as a whole but only with preordered products, we're checking here
		 * if we're only having preordered products whether we're on the 1st or 4th option as
		 * both are treated as whole orders, the difference being having mixed products with the first option.
		 */
		if ( $this->orderHasOnlyPreorderedProducts( $order, $this->cart ) ) {
			$this->emailIds['preorderIds'][] = $orderId;
		}

		// main action firing emails.
		do_action( 'preorder_email', $this->emailIds );
	}

	/**
	 * @param $orderObj
	 * @param $cartObj
	 */
	public function orderHasOnlyPreorderedProducts( $orderObj, $cartObj ) {
		return count( $orderObj->get_items() ) === count( $cartObj->getPreOrderProducts() );
	}

	/**
	 * @param $order_id
	 */
	public function checkWholeOrders( $order_id ) {
		/*if ( get_post_meta( $order_id, '_pre_order_date', true ) ) {
			$order = wc_get_order( $order_id );
			$order->set_status( 'wc-pre-ordered' );
			$order->save();
		}*/

		$order = wc_get_order( $order_id );
		if ( $order->get_meta('_pre_order_date') ){
			$order->set_status( 'wc-pre-ordered' );
			$order->save();
		}
	}

	/**
	 * @param  $order
	 * @return mixed
	 */
	private function getPreorderedProducts( $order ) {
		$preorderedProducts = [];
		foreach ( $order->get_items() as $item ) {
			$productId  = 0 !== $item->get_variation_id() ? $item->get_variation_id() : $item->get_product_id();
			$isPreOrder = get_post_meta( $productId, '_pre_order_date', true );
			if ( $isPreOrder && strtotime( $isPreOrder ) > time() ) {
				$preorderedProducts[] = $item;
			}
		}

		return $preorderedProducts;
	}

	/**
	 * Get the shipping total and split it into the amount of orders generated
	 * @param  $order
	 * @return int
	 */
	private function getSplitShippingFactor( $order ) {
		if ( 'yes' !== get_option( 'wc_preorders_multiply_shipping' ) ) {
			return 1;
		}

		// If we are working on partial mode, then we will split it in 2 halves
		if ( 'partial' === $this->preordersMode ) {
			return 2;
			// Otherwise we will have to split it by the amount of orders that we have
		}
		if ( 'individual' === $this->preordersMode ) {
			return 1+\count( $this->getPreorderedProducts( $order ) );
		}

		return 1;
	}

	/**
	 * @param  $prefix
	 * @param  $fields
	 * @return mixed
	 */
	private function getFilteredFields( $prefix, $fields ) {
		return $this->stripFieldsPrefix( $prefix, $this->filterFields( $prefix, $fields ) );
	}

	/**
	 * @param $prefix
	 * @param $fields
	 */
	private function stripFieldsPrefix( $prefix, $fields ) {
		return array_combine(
			array_map(
				function ( $k ) use ( $prefix ) {
					return str_replace( $prefix, '', $k );
				},
				array_keys( $fields )
			),
			array_values( $fields )
		);
	}

	/**
	 * @param  $prefix
	 * @param  $fields
	 * @return int
	 */
	private function filterFields( $prefix, $fields ) {
		return array_filter( $fields, function ( $key ) use ( $prefix ) {
			return 0 === strpos( $key, $prefix );
		}, ARRAY_FILTER_USE_KEY );
	}
}
