<?php
namespace Aventa;

use Elementor\Controls_Manager;
use Elementor\Widget_Base;

if ( ! defined( "ABSPATH" ) ) {
	exit; // Exit if accessed directly
}

class Rad_Products_Widget extends Widget_Base {

	public function get_name() {
		return "rad-products";
	}

	public function get_title() {
		return esc_html__( "Products", "aventa" );
	}

	public function get_icon() {
		return "eicon-products";
	}

	public function get_keywords() {
		return [ "woocommerce", "shop", "store", "product", "archive", "aventa" ];
	}

	public function get_categories() {
		return [ Rad_Helpers::WC_WIDGETS_CATEGORY ];
	}

	protected function register_controls() {

    $this->start_controls_section(
			"section_query",
			[
				"label" => esc_html__( "Products", "aventa" ),
				"tab" => Controls_Manager::TAB_CONTENT,
			]
		);

    $this->add_control(
			"products_query",
			[
				"label" => esc_html__( "Products Query", "aventa" ),
				"type" => Controls_Manager::SELECT,
				"default" => "products",
				"options" => [
          "products"              => esc_html__("Manual Selection", "aventa"),
          "recent_products"       => esc_html__("Latest", "aventa"),
          "sale_products"         => esc_html__("On-Sale", "aventa"),
          "featured_products"     => esc_html__("Featured", "aventa"),
          "top_rated_products"    => esc_html__("Top-Rated", "aventa"),
          "best_selling_products" => esc_html__("Best-Selling", "aventa"),
				],
			]
		);

    $this->add_control(
			"custom_products",
			[
				"label"         => esc_html__( "Select Products", "aventa" ),
				"type"          => Controls_Manager::SELECT2,
				"multiple"      => true,
				"options"       => Rad_Helpers::get_products(),
        "default"       => "",
				"condition"     => [
          "products_query" => "products"
        ]
			]
		);

		$this->add_control(
			"divider",
			[ "type" => Controls_Manager::DIVIDER ]
		);

    $this->add_control(
			"custom_categories",
			[
				"label"         => esc_html__( "Categories", "aventa" ),
				"type"          => Controls_Manager::SELECT2,
				"multiple"      => true,
				"options"       => Rad_Helpers::get_wc_categories()
			]
		);

    $this->add_control(
			"custom_tags",
			[
				"label"         => esc_html__( "Tags", "aventa" ),
				"type"          => Controls_Manager::SELECT2,
				"multiple"      => true,
				"options"       => Rad_Helpers::get_wc_tags()
			]
		);

		$this->add_control(
			"terms_include",
			[
				"label" 				=> esc_html__( "Include/Exclude", "aventa" ),
				"type" 					=> Controls_Manager::SWITCHER,
				"label_on" 			=> esc_html__( "In", "aventa" ),
				"label_off" 		=> esc_html__( "Ex", "aventa" ),
				"return_value" 	=> "include",
				"default" 			=> "include",
				"description"   => "Include or exclude selected categories and tags?"
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			"section_settings",
			[
				"label" => esc_html__( "Settings", "aventa" ),
			]
		);

		$this->add_control(
			"product_columns",
			[
				"label" => esc_html__( "Columns", "aventa" ),
				"type" => Controls_Manager::NUMBER,
				"min" => 1,
				"max" => 6,
				"default" => 4,
				"required" => true
			]
		);

		$this->add_control(
			"products_limit",
			[
				"label" => esc_html__( "Max Product Number", "aventa" ),
				"type" => Controls_Manager::NUMBER,
				"min" => 1
			]
		);

		$this->add_control(
			"products_paginate",
			[
				"label" 				=> esc_html__( "Pagination", "aventa" ),
				"type" 					=> Controls_Manager::SWITCHER,
				"label_on" 			=> esc_html__( "On", "aventa" ),
				"label_off" 		=> esc_html__( "Off", "aventa" ),
				"return_value" 	=> "true",
				"condition" => [
					"products_display_mode" => "grid"
				]
			]
		);

		$this->add_control(
			"products_order_by",
			[
				"label"         => esc_html__( "Order By", "aventa" ),
				"type"          => Controls_Manager::SELECT,
				"options"       => [
					"date"								=> esc_html__( "Date", "aventa" ),
					"ID"									=> esc_html__( "ID", "aventa" ),
					"author"							=> esc_html__( "Author", "aventa" ),
					"title"								=> esc_html__( "Title", "aventa" ),
					"modified"						=> esc_html__( "Date Modified", "aventa" ),
					"rand"								=> esc_html__( "Random", "aventa" ),
					"comment_count"				=> esc_html__( "Number of Comments", "aventa" ),
					"menu_order"				  => esc_html__( "Menu Order", "aventa" ),
					"menu_order title"  	=> esc_html__( "Menu Order & Title", "aventa" ),
					"include"							=> esc_html__( "Include", "aventa" ),
					"post__in"					 	=> esc_html__( "Custom Post-In", "aventa" )
				],
				"description" 	=> "More at <a href='https://developer.wordpress.org/reference/classes/wp_query/#order-orderby-parameters' target='_blank'>WordPress Codex</a> page.",
        "default"       => "date",
			]
		);

		$this->add_control(
			"products_sort_order",
			[
				"label"         => esc_html__( "Sort Order", "aventa" ),
				"type"          => Controls_Manager::SELECT,
				"options"       => [
					"ASC" 	=> esc_html__( "Ascending", "aventa" ),
					"DESC" 	=> esc_html__( "Descending", "aventa" ),
				],
        "default"       => "DESC",
			]
		);
		
		$this->end_controls_section();

		// General style
    $this->start_controls_section(
			"section_general_style",
			[
				"label" => esc_html__( "General", "aventa" ),
				"tab" => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			"products_display_mode",
			[
				"label"         => esc_html__( "Display Mode", "aventa" ),
				"type"          => Controls_Manager::SELECT,
				"options"       => [
					"grid" 	=> esc_html__( "Grid", "aventa" ),
					"carousel" 	=> esc_html__( "Carousel", "aventa" ),
				],
        "default"       => "grid",
				"required"			=> true
			]
		);

		$this->add_control(
			"products_grid_mode",
			[
				"label"         => esc_html__( "Grid Mode", "aventa" ),
				"type"          => Controls_Manager::SELECT,
				"options"       => [
					"general" 	=> esc_html__( "General", "aventa" ),
					"sticky" 		=> esc_html__( "Sticky", "aventa" ),
					"scatter" 	=> esc_html__( "Scatter", "aventa" ),
				],
        "default"       => "general",
				"condition" => [
					"products_display_mode" => "grid"
				]
			]
		);

		$this->add_control(
			"products_style",
			[
				"label"         => esc_html__( "Style", "aventa" ),
				"type"          => Controls_Manager::SELECT,
				"options"       => [
					"general" 	=> esc_html__( "General", "aventa" ),
					"creative" 	=> esc_html__( "Creative", "aventa" ),
					"bt-hover" 	=> esc_html__( "Buttons on Hover", "aventa" ),
					"qv-hover" 	=> esc_html__( "Quickview on Hover", "aventa" ),
				],
				"required" 		=> true,
        "default"     => "bt-hover"
			]
		);

		$this->add_control(
			"products_navigation",
			[
				"label"         => esc_html__( "Navigation Type", "aventa" ),
				"type"          => Controls_Manager::SELECT,
				"options"       => [
					"next-prev" 	=> esc_html__( "Next/Prev Button", "aventa" ),
					"scrollbar" 	=> esc_html__( "Scrollbar", "aventa" ),
					"pagination" 	=> esc_html__( "Bullets", "aventa" ),
				],
        "default"       => "next-prev",
				"condition"     => [
					"products_display_mode" => "carousel"
				]
			]
		);

		$this->add_control(
			"products_navigation_style",
			[
				"label"         => esc_html__( "Navigation Style", "aventa" ),
				"type"          => Controls_Manager::SELECT,
				"options"       => [
					"dark" 	=> esc_html__( "Dark", "aventa" ),
					"light" 	=> esc_html__( "Light", "aventa" ),
				],
        "default"       => "dark",
				"condition"     => [
					"products_display_mode" => "carousel"
				]
			]
		);
		
		$this->add_control(
			"products_gutter",
			[
				"label"         => esc_html__( "Gutter", "aventa" ),
				"type"          => Controls_Manager::SELECT,
				"options"       => [
					"0" 	=> esc_html__( "One Unit", "aventa" ),
					"1" 	=> esc_html__( "Two Units", "aventa" ),
				],
        "default"       => "0"
			]
		);

		$this->add_control(
			"products_autoplay",
			[
				"label" 				=> esc_html__( "Autoplay", "aventa" ),
				"type" 					=> Controls_Manager::SWITCHER,
				"label_on" 			=> esc_html__( "On", "aventa" ),
				"label_off" 		=> esc_html__( "Off", "aventa" ),
				"return_value" 	=> "on",
				"default" 			=> "on",
				"condition"     => [
					"products_display_mode" => "carousel"
				]
			]
		);

		$this->add_control(
			"products_hover_image",
			[
				"label" 				=> esc_html__( "Hover Image", "aventa" ),
				"type" 					=> Controls_Manager::SWITCHER,
				"label_on" 			=> esc_html__( "Show", "aventa" ),
				"label_off" 		=> esc_html__( "Hide", "aventa" ),
				"return_value" 	=> "show",
				"default" 			=> "show"
			]
		);

		$this->add_control(
			"products_quickview_button",
			[
				"label" 				=> esc_html__( "Quick-view Button", "aventa" ),
				"type" 					=> Controls_Manager::SWITCHER,
				"label_on" 			=> esc_html__( "Show", "aventa" ),
				"label_off" 		=> esc_html__( "Hide", "aventa" ),
				"return_value" 	=> "enable",
				"default" 			=> "enable",
				"condition"     => [
					"products_style!" => "qv-hover"
				]
			]
		);

		$this->add_control(
			"products_wishlist_button",
			[
				"label" 				=> esc_html__( "Wishlist Button", "aventa" ),
				"type" 					=> Controls_Manager::SWITCHER,
				"label_on" 			=> esc_html__( "Show", "aventa" ),
				"label_off" 		=> esc_html__( "Hide", "aventa" ),
				"return_value" 	=> "enable",
				"default" 			=> "enable"
			]
		);
		
		$this->end_controls_section();

		$this->start_controls_section(
			"section_image_style",
			[
				"label" => esc_html__( "Image", "aventa" ),
				"tab" => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
      "products_image_size",
      [
        "label" => esc_html__( "Image Size", "aventa" ),
        "type" => Controls_Manager::SELECT,
        "options" => [
          "full"   								=> esc_html__( "Full", "aventa" ),
          "woocommerce_single"   	=> esc_html__( "Single Product Image", "aventa" ),
          "woocommerce_thumbnail" => esc_html__( "Product Thumbnail", "aventa" ),
          "custom"   							=> esc_html__( "Custom", "aventa" ),
				],
        "default" => "woocommerce_single"
      ]
    );

    $this->add_control(
			"products_image_dimension",
			[
				"label" => esc_html__( "Image Dimension", "aventa" ),
				"type" => Controls_Manager::IMAGE_DIMENSIONS,
				"description" => esc_html__( "Crop the original image size to any custom size.", "aventa" ),
        "condition" => [
          "products_image_size" => "custom"
        ]
			]
		);

		$this->add_control(
			"products_crop_image",
			[
				"label" 				=> esc_html__( "Crop Image?", "aventa" ),
				"type" 					=> Controls_Manager::SWITCHER,
				"label_on" 			=> esc_html__( "Yes", "aventa" ),
				"label_off" 		=> esc_html__( "No", "aventa" ),
				"return_value" 	=> "true",
				"default" 			=> "true",
				"condition" => [
          "products_image_size" => "custom"
        ]
			]
		);

		$this->end_controls_section();

	}

	protected function render() {

		$settings = $this->get_settings();

		$query = esc_html( $settings[ "products_query" ] );

		$atts = [];
		$atts[ "columns" ] 				= esc_attr( $settings[ "product_columns" ] );
		$atts[ "orderby" ] 				= esc_attr( $settings[ "products_order_by" ] );
		$atts[ "order" ] 					= esc_attr( $settings[ "products_sort_order" ] );
		$atts[ "limit" ] 					= esc_attr( $settings[ "products_limit" ] );
		$atts[ "layout" ] 				= esc_attr( $settings[ "products_display_mode" ] );
		$atts[ "style" ] 					= esc_attr( $settings[ "products_style" ] );
    $atts[ "gutter" ]   			= esc_attr( $settings[ "products_gutter" ] );
    $atts[ "navigation" ]    	= esc_attr( $settings[ "products_navigation" ] );
		$atts[ "nav_style" ] 			= esc_attr( $settings[ "products_navigation_style" ] );
		$atts[ "grid_mode" ] 			= esc_attr( $settings[ "products_grid_mode" ] );
		$atts[ "wishlist" ] 			= esc_attr( $settings[ "products_wishlist_button" ] );
		$atts[ "image_size" ]     = esc_attr( $settings[ "products_image_size" ] );

		if ( $settings[ "custom_products" ] ) {
			$atts[ "ids" ] = implode( ", ", $settings[ "custom_products" ] );
		}

		if ( $settings[ "custom_categories" ] ) {
			$atts[ "category" ] = implode( ", ", $settings[ "custom_categories" ] );
		}

		if ( $settings[ "custom_tags" ] ) {
			$atts[ "tag" ] = implode( ", ", $settings[ "custom_tags" ] );
		}

		if ( !$settings[ "terms_include" ] ) {
			$atts[ "cat_operator" ] = "NOT IN";
			$atts[ "tag_operator" ] = "NOT IN";
		} 

		if ( $atts[ "layout" ] == "grid" && $settings[ "products_paginate" ] == "true" ) {
      $atts[ "paginate" ] = true;
		}

		if ( $atts[ "layout" ] == "carousel" && $settings[ "products_autoplay" ] == "on" ) {
      $atts[ "is_autoplay" ] = $settings[ "products_autoplay" ];
		}

		if ( $atts[ "style" ] != "qv-hover" && $settings[ "products_quickview_button" ] != "enable" ) {
			$atts[ "quickview" ] = "disable";
		}

		if ( $atts[ "image_size" ] == "custom" ) {
			$atts[ "image_size_width" ]     = esc_attr( $settings[ "products_image_dimension" ][ "width" ] );
      $atts[ "image_size_height" ]   	= esc_attr( $settings[ "products_image_dimension" ][ "height" ] );
      $atts[ "image_size_crop" ]      = esc_attr( $settings[ "products_crop_image" ] );
		}

		$html = "";

		if ( $query == "products" ) 
		{
			$html = rad_products( $atts );
		} 
		else if ( $query == "featured_products" ) 
		{
			$html = rad_featured_products( $atts );
		}
		else if ( $query == "best_selling_products" ) 
		{
			$html = rad_best_selling_products( $atts );
		}
		else if ( $query == "top_rated_products" ) 
		{
			$html = rad_top_rated_products( $atts );
		}
		else if ( $query == "sale_products" ) 
		{
			$html = rad_sale_products( $atts );
		}
		else if ( $query == "recent_products" ) 
		{
			$html = rad_recent_products( $atts );
		}

    echo "$html";
	}

}
