<?php

namespace Aventa;

use Elementor\Utils;
use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Repeater;


if ( !defined( "ABSPATH" ) ) exit; 

class Rad_Gallery_Carousel_Widget extends Widget_Base{

  public function get_name() {
    return "rad-gallery-carousel";
  }

  public function get_title() {
    return esc_html__("Gallery Carousel", "aventa");
  }

  public function get_keywords() {
		return [ "aventa" ];
	}

  public function get_icon() {
    return "eicon-post-slider";
  }

  public function get_categories() {
    return [ Rad_Helpers::WIDGETS_CATEGORY ];
  }

  protected function register_controls() {

		// Widget content
    $this->start_controls_section(
			"section_title",
			[
				"label" => esc_html__( "Gallery Carousel", "aventa" ),
			]
		);

    $gallery_item = new Repeater();

    $gallery_item->add_control(
			"gc_image",
			[
				"label" => esc_html__( "Image", "aventa" ),
				"type" => Controls_Manager::MEDIA,
				"default" => [
					"url" => Utils::get_placeholder_image_src(),
				]
			]
		); 

		$gallery_item->add_control(
			"gc_title", [
				"label" => esc_html__( "Title", "aventa" ),
				"type" => Controls_Manager::TEXT,
			]
		);

		$gallery_item->add_control(
			"gc_subtitle", [
				"label" => esc_html__( "Subtitle", "aventa" ),
				"type" => Controls_Manager::TEXT,
			]
		);

    $gallery_item->add_control(
      "gc_has_link",
      [
        "label" => esc_html__( "Has Link?", "aventa" ),
        "type" => Controls_Manager::SWITCHER,
        "label_on" => esc_html__( "Yes", "aventa" ),
        "label_off" => esc_html__( "No", "aventa" ),
        "return_value" => "true"
      ]
    );

    Rad_Helpers::link( $gallery_item, "gc_link", "Link Text", "Link URL", [ "gc_has_link" => "true" ] );

    $this->add_control(
			"gc_items",
			[
				"label" => esc_html__( "Gallery Carousel", "aventa" ),
				"type" => Controls_Manager::REPEATER,
				"fields" => $gallery_item->get_controls(),
				"title_field" => "{{{ gc_title }}}"
			]
		);

		$this->end_controls_section();

    // General style
    $this->start_controls_section(
			"section_general_style",
			[
				"label" => esc_html__( "General", "aventa" ),
				"tab" => Controls_Manager::TAB_STYLE,
			]
		);

    $this->add_control(
      "gc_visible_items",
      [
        "label" => esc_html__( "Visible Items", "aventa" ),
        "type" => Controls_Manager::SELECT,
        "options" => [
          "1" => 1,
          "2" => 2,
          "3" => 3,
          "4" => 4,
          "5" => 5,
          "6" => 6,
          "7" => 7,
          "8" => 8
        ],
        "default" => "1"
      ]
    );

    $this->add_control(
      "gc_nav_type",
      [
        "label" => esc_html__( "Navigation Type", "aventa" ),
        "type" => Controls_Manager::SELECT,
        "options" => [
          "next-prev"   => esc_html__( "Next/Prev Buttons", "aventa" ),
          "scrollbar"   => esc_html__( "Scrollbar", "aventa" ),
          "bullets"  => esc_html__( "Bullets", "aventa" ),
        ],
        "default" => "next-prev"
      ]
    );

    $this->add_control(
      "gc_nav_style",
      [
        "label" => esc_html__( "Navigation Style", "aventa" ),
        "type" => Controls_Manager::CHOOSE,
        "options" => [
          "light" => [
            "title" => esc_html__( "Light", "aventa" ),
            "icon" => "fas fa-sun"
          ],
          "dark" => [
            "title" => esc_html__( "Dark", "aventa" ),
            "icon" => "fas fa-moon"
          ]
        ],
        "default" => "dark"
      ]
    );

    $this->add_control(
      "gc_overlay_color",
      [
        "label" => esc_html__( "Overlay Color", "aventa" ),
				"type" => Controls_Manager::COLOR,
				"selectors" => [
					"{{WRAPPER}} .gallery-carousel-block .overlay" => "background-color: {{VALUE}};",
				],
      ]
    );
    
    $this->add_control(
      "gc_freemode",
      [
        "label" => esc_html__( "Freemode", "aventa" ),
        "description" => esc_html__( "Freemode: Slides won't have fixed positions.", "aventa" ),
        "type" => Controls_Manager::SWITCHER,
        "label_on" => esc_html__( "Yes", "aventa" ),
        "label_off" => esc_html__( "No", "aventa" ),
        "return_value" => "true"
      ]
    );

    $this->add_control(
      "gc_autoplay",
      [
        "label" => esc_html__( "Autoplay", "aventa" ),
        "type" => Controls_Manager::SWITCHER,
        "label_on" => esc_html__( "Yes", "aventa" ),
        "label_off" => esc_html__( "No", "aventa" ),
        "return_value" => "true",
      ]
    );

    $this->add_control(
      "gc_gutter",
      [
        "label" => esc_html__( "Gutter", "aventa" ),
        "type" => Controls_Manager::SWITCHER,
        "label_on" => esc_html__( "Yes", "aventa" ),
        "label_off" => esc_html__( "No", "aventa" ),
        "default" => "true",
        "return_value" => "true"
      ]
    );

    $this->add_control(
      "gc_image_size",
      [
        "label" => esc_html__( "Image Size", "aventa" ),
        "type" => Controls_Manager::SELECT,
        "options" => [
          "full"   => esc_html__( "Full", "aventa" ),
          "medium"   => esc_html__( "Medium", "aventa" ),
          "thumbnail"   => esc_html__( "Thumbnail", "aventa" ),
          "custom"   => esc_html__( "Custom", "aventa" ),
        ],
        "separator" => "before",
        "default" => "full"
      ]
    );

    $this->add_control(
			"gc_image_dimension",
			[
				"label" => esc_html__( "Image Dimension", "aventa" ),
				"type" => Controls_Manager::IMAGE_DIMENSIONS,
				"description" => esc_html__( "Crop the original image size to any custom size.", "aventa" ),
        'default' => [
          'width' => '',
          'height' => '',
        ],
        "condition" => [
          "gc_image_size" => "custom"
        ]
			]
		);

    $this->add_control(
			"gc_animation",
			[
				"label" => esc_html__( "Animation", "aventa" ),
				"type" => Controls_Manager::SELECT,
				"default" => "img-anim-zoom-in",
				"options" => [
          "img-anim-zoom-in" => esc_html__( "Zoom-In", "aventa" ),
          "img-anim-mask" => esc_html__( "Mask", "aventa" ),
          "" => esc_html__( "None", "aventa" )
				],
        "separator" => "before"
			]
		);

		$this->end_controls_section();

    $this->start_controls_section(
			"section_title_style",
			[
				"label" => esc_html__( "Title", "aventa" ),
				"tab" => Controls_Manager::TAB_STYLE,
			]
		);

    Rad_Helpers::typography( $this, "gc_title", ".gallery-carousel-block .text-wrapper .text" );

		$this->end_controls_section();

    $this->start_controls_section(
			"section_subtitle_style",
			[
				"label" => esc_html__( "Subtitle", "aventa" ),
				"tab" => Controls_Manager::TAB_STYLE,
			]
		);

    Rad_Helpers::typography( 
      $this, 
      "gc_subtitle", 
      ".gallery-carousel-block .text-wrapper .subtext", 
      [ "min" => 10, "max" => 40, "step" => 1 ],
      15
    );

		$this->end_controls_section();


  }

  protected function render() {
    $settings = $this->get_settings_for_display();

		if ( !$settings[ "gc_items" ] ) {
			return;
		}

    $classes = [ 
      "gallery-carousel-block", 
      "resp-block",
      "carousel"
    ];

    // User inputs
		$gallery_items           = $settings[ "gc_items" ];
		$visible_items    = esc_attr( $settings[ "gc_visible_items" ] );
		$nav_type         = esc_attr( $settings[ "gc_nav_type" ] );
		$nav_style         = esc_attr( $settings[ "gc_nav_style" ] );
		$freemode         = esc_attr( $settings[ "gc_freemode" ] );
		$is_autoplay      = $settings[ "gc_autoplay" ] ? "on" : "off";
		$gutter           = esc_attr( $settings[ "gc_gutter" ] );
		$image_size       = esc_attr( $settings[ "gc_image_size" ] );
    $image_width      = "";
    if(isset($settings[ "gc_image_dimension" ][ "width" ]))
    {
      $image_width      = esc_attr( $settings[ "gc_image_dimension" ][ "width" ] );
    }

    $image_height      = "";
    if(isset($settings[ "gc_image_dimension" ][ "height" ]))
    {
      $image_height      = esc_attr( $settings[ "gc_image_dimension" ][ "height" ] );
    }

    $animation        = esc_attr( $settings[ "gc_animation" ] );
    $image_crop       = ( empty( $image_dimension[ "width" ] ) || empty( $image_dimension[ "height" ] ) )
      ? true
      : false ;

    // Assign proper class based on selected options
    $classes[] = $nav_style;
    $classes[] = $freemode      ?   "freemode"            : "";
    $classes[] = !$gutter       ?   "no-gutter"           : "";
    $classes[] = $animation     ?   "img-anim $animation" : "";

    if ( $gallery_items ) {

      // Attributes
      $visible_attr  = "data-visibleitems='$visible_items'";
      $autoplay_attr = "data-autoplay='$is_autoplay'";

      // HTML
      $html = "";
      $html .= rad_make_container("div", "", "", implode( " ",  $classes ), "", $autoplay_attr, false);
      $html .= rad_make_container("div", "", "", "swiper-container", "", $visible_attr, false);
      $html .= rad_make_container("div", "", "", "swiper-wrapper", "", "", false);

      foreach ( $gallery_items as $gallery_item ) {

        $title      = esc_html( $gallery_item[ "gc_title" ] );
        $subtitle   = esc_html( $gallery_item[ "gc_subtitle" ] );
        $link_text = $link_url = "";
        if ( $gallery_item[ "gc_has_link" ] ) {
          $link_text  = esc_html( $gallery_item[ "gc_link_title" ] );
          $link_url   = esc_url( $gallery_item[ "gc_link_url" ]["url"] );
        }

        $image = rad_get_image( 
          $gallery_item[ "gc_image" ][ "id" ], 
          "", 
          $title, 
          $image_size, 
          $image_width,
          $image_height,
          $image_crop,
          true
        );

        if ($image == "") {
          continue;
        }

        $style        = "padding-top:" . rad_get_height_percentage($image) . "%;";
        $slider_style = "";

        if ( $freemode ) {
          $slider_style = "width:" . rad_get_image_width($image) . "px;";
          $slider_style .= "max-width:40vw;";
        }

        $html .= rad_make_container("div", "", "", "swiper-slide item carousel_item", $slider_style, "", false);
        $html .= rad_make_container("div", "", "", "img img-container lazy-load lazy-load-on-load", $style, "", false);
        $html .= rad_make_container("div", "", "", "img-wrap", "", "", false);
        $html .= $image;
        $html .= "<div class='overlay'></div>";
        if ( $gallery_item[ "gc_has_link" ] ) {
          $html .= "<a href='$link_url' target='_blank' title='$link_text'>$link_text</a>";
        }
        $html .= "
          <span class='text-wrapper'>
            <span class='text'>$title</span>
            <span class='subtext'>$subtitle</span>
          </span>";
        $html .= "</div>";
        $html .= "</div>";
        $html .= "</div>";
      }

      $html .= "</div></div>";

      if ( $nav_type == "next-prev" ) 
      {
        $html .= rad_make_container("div", "", "", "rad-carousel-next unselectable", "", "");
        $html .= rad_make_container("div", "", "", "rad-carousel-prev unselectable", "", "");
      } 
      elseif ( $nav_type == "scrollbar" ) 
      {
        $html .= rad_make_container("div", "", "", "swiper-scrollbar", "", "");
      } 
      else {
        $html .= rad_make_container("div", "", "", "swiper-pagination", "", "");
      }

      $html .= "</div>";

      echo "$html";
    }

  }

}