<?php

namespace Aventa;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Utils;

if ( !defined( "ABSPATH" ) ) exit; 


class Rad_Banner_Widget extends Widget_Base{

  public function get_name() {
    return "rad-banner";
  }

  public function get_title() {
    return esc_html__("Banner", "aventa");
  }

  public function get_keywords() {
		return [ "aventa" ];
	}

  public function get_icon() {
    return "eicon-info-box";
  }

  public function get_categories() {
    return [ Rad_Helpers::WIDGETS_CATEGORY ];
  }

  protected function register_controls() {
    $this->start_controls_section(
			"section_title",
			[
				"label" => esc_html__( "Banner", "aventa" ),
			]
		);

		$this->add_control(
			"banner_type",
			[
				"label" => esc_html__( "Display Mode", "aventa" ),
				"type" => Rad_Visual_Option_Selector_Control::TYPE,
				"default" => "general-style",
        "separator" => "after",
        "icon_dir" => "banner_type",
				"options" => [
          "general-style" => "general",
          "hover-style" => "hover",
          "creative-style" => "creative"
				],
			]
		);

    // Banner image
    $this->add_control(
      "banner_image",
      [
        "label" => esc_html__( "Image", "aventa" ),
        "type" => Controls_Manager::MEDIA,
        "default" => [
          "url" => Utils::get_placeholder_image_src(),
        ]
      ]
    ); 

    // Banner thumb
    $this->add_control(
      "banner_thumb",
      [
        "label" => esc_html__( "Thumbnail", "aventa" ),
        "type" => Controls_Manager::MEDIA,
        "default" => [
          "url" => Utils::get_placeholder_image_src(),
        ], 
        "condition" => [
          "banner_type" => "creative-style"
        ]
      ]
    ); 

    // Separator
    $this->add_control(
			"hr",
			[ "type" => Controls_Manager::DIVIDER ]
		);

		// Banner title
		$this->add_control(
			"banner_title",
			[
				"label" => esc_html__( "Title", "aventa" ),
				"type" => Controls_Manager::TEXTAREA,
				"default" => esc_html__( "Title", "aventa" ),
			]
		);

		// Banner subtitle
		$this->add_control(
			"banner_subtitle",
			[
				"label" => esc_html__( "Subtitle", "aventa" ),
				"type" => Controls_Manager::TEXTAREA,
				"default" => esc_html__( "Subtitle", "aventa" ),
				"separator" => "before"
			]
		);

		// Banner text
		$this->add_control(
			"banner_text",
			[
				"label" => esc_html__( "Text", "aventa" ),
				"type" => Controls_Manager::TEXTAREA,
				"default" => esc_html__( "Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt.", "aventa" ),
				"separator" => "before"
			]
		);

    // Separator
    $this->add_control(
			"hr_second",
			[ "type" => Controls_Manager::DIVIDER ]
		);

    $this->add_control(
      "banner_has_link",
      [
        "label" => esc_html__( "Has Link?", "aventa" ),
        "type" => Controls_Manager::SWITCHER,
        "label_on" => esc_html__( "Yes", "aventa" ),
        "label_off" => esc_html__( "No", "aventa" ),
        "return_value" => "true",
      ]
    );

    // Banner link
    Rad_Helpers::link( 
      $this, 
      "banner_link", 
      esc_html__( "Link Text", "aventa" ),  
      esc_html__( "Link URL", "aventa" ), 
      [ "banner_has_link" => "true" ]
    );
		
		$this->end_controls_section();

    // General Settings
    $this->start_controls_section(
			"section_general_style",
			[
				"label" => esc_html__( "General", "aventa" ),
				"tab" => Controls_Manager::TAB_STYLE,
			]
		);

    // Banner alignment
		Rad_Helpers::alignment( $this, "banner_align", [], [ "left", "right" ], "left", false );

    // Hover overlay color
    $this->add_control(
			"banner_overlay_color",
			[
				"label" => esc_html__( "Overlay Color", "aventa" ),
				"type" => Controls_Manager::COLOR,
				"default" => "#000",
				"selectors" => [
					"{{WRAPPER}} .banner-block .overlay" => "background-color: {{VALUE}};",
				],
        "condition" => [
          "banner_type" => "hover-style"
        ]
			]
		);

    // Image animation
    $this->add_control(
			"banner_animation",
			[
				"label" => esc_html__( "Animation", "aventa" ),
				"type" => Controls_Manager::SELECT,
				"default" => "img-anim-zoom-in",
				"options" => [
          "img-anim-zoom-in" => esc_html__( "Zoom-In", "aventa" ),
          "img-anim-mask" => esc_html__( "Mask", "aventa" ),
          "" => esc_html__( "None", "aventa" )
				],
			]
		);

    // Animation delay
    $this->add_control(
			"banner_animation_delay",
			[
				"label" => esc_html__( "Animation Delay (ms)", "aventa" ),
				"type" => Controls_Manager::NUMBER,
				"default" => 0,
        "separator" => "after",
        "condition" => [
          "banner_animation!" => ""
        ]
			]
		);

		$this->end_controls_section();

		// Title styles
		$this->start_controls_section(
			"section_title_style",
			[
				"label" => esc_html__( "Title", "aventa" ),
				"tab" => Controls_Manager::TAB_STYLE,
			]
		);

		Rad_Helpers::heading_tag( $this, "banner_title_tag" );

		Rad_Helpers::typography( $this, "banner_title_style", ".banner-block .title" );

		$this->end_controls_section();


    // Subtitle styles
    $this->start_controls_section(
			"section_subtitle_style",
			[
				"label" => esc_html__( "Subtitle", "aventa" ),
				"tab" => Controls_Manager::TAB_STYLE,
			]
		);

		Rad_Helpers::typography( 
			$this, 
			"banner_subtitle_style", 
			".banner-block .subtitle", 
			[ "min" => 10, "max" => 40, "step" => 1 ], 
			15 
		);

		$this->end_controls_section();

    // Text styles
    $this->start_controls_section(
			"section_text_style",
			[
				"label" => esc_html__( "Text", "aventa" ),
				"tab" => Controls_Manager::TAB_STYLE,
			]
		);

		Rad_Helpers::typography( 
			$this, 
			"banner_text_style", 
			".banner-block .text p", 
			[ "min" => 10, "max" => 40, "step" => 1 ], 
			15 
		);

		$this->end_controls_section();

    // Link style
    $this->start_controls_section(
      "section_link_style",
      [
        "label" => esc_html__( "Link", "aventa" ),
        "tab" => Controls_Manager::TAB_STYLE,
      ]
    );

    Rad_Helpers::typography( 
			$this, 
			"banner_link_style", 
			".banner-block a", 
			[ "min" => 10, "max" => 40, "step" => 1 ], 
			15 
		);

    $this->end_controls_section();
  }
  

  protected function render() {
    $settings = $this->get_settings_for_display();

		if ( 
      "" === $settings[ "banner_title" ]      && 
      "" === $settings[ "banner_subtitle" ]   && 
      "" === $settings[ "banner_text" ]       && 
      "" === $settings[ "banner_image" ] 
    ) {
			return;
		}

    $classes = [ 
      "banner-block"
    ];

    // User inputs
		$type               = esc_html( $settings[ "banner_type" ] );
		$title              = esc_html( $settings[ "banner_title" ] );
    $subtitle           = esc_html( $settings[ "banner_subtitle" ] );
		$text               = esc_html( $settings[ "banner_text" ] );
		$alignment          = esc_attr( $settings[ "banner_align" ] );
		$link_text          = isset( $settings[ "banner_link_title" ] ) ? esc_html( $settings[ "banner_link_title" ] ) : "";;
		$url                = isset( $settings[ "banner_link_url" ]["url"] ) ? esc_url( $settings[ "banner_link_url" ]["url"] ) : "";
		$animation          = esc_attr( $settings[ "banner_animation" ] );
		$animation_delay    = esc_attr( $settings[ "banner_animation_delay" ] );

    $image = $settings[ "banner_image" ][ "id" ]
      ? wp_get_attachment_image( 
          esc_html( $settings[ "banner_image" ][ "id" ] ), 
          "full", 
          false, 
          [ "class" => "main-img", "alt" => $title ] 
        )
      : Rad_Helpers::get_placeholder_image();
    $thumb = isset($settings[ "banner_thumb" ][ "id" ])
      ? wp_get_attachment_image( 
        esc_html( $settings[ "banner_thumb" ][ "id" ] ), 
        "full", 
        false, 
        [ "alt" => $title ] 
      )
      : Rad_Helpers::get_placeholder_image();

		// Assign class based on selected options
    $classes[] = $type;
    $classes[] = $alignment;
    if ( $settings["banner_animation"] ) {
      $classes[] = "img-anim $animation";
    }

		// Elements attributes
		$this->add_render_attribute( "banner", "class", implode( " ", array_map( "esc_attr", $classes ) ) );
		$this->add_render_attribute( "banner", "data-delay", $animation_delay );
		$this->add_render_attribute( "title", "class", "title" );
		$this->add_render_attribute( "subtitle", "class", "subtitle" );
		$this->add_render_attribute( "content_container", "class", "text" );
		$this->add_render_attribute( "link", "class", "underline-style init-show" );
		$this->add_render_attribute( "link", "href", $url );

		// HTML elements
    $image_html = "
        <div class='img-container target-img-anim' >
          <div class='img-wrap' > $image </div>
        </div>
      ";
    
    $title_html = sprintf( 
      '<%1$s %2$s>%3$s</%1$s>', 
      Utils::validate_html_tag( $settings[ "banner_title_tag" ] ), 
      $this->get_render_attribute_string( "title" ), 
      $title 
    );

    $subtitle_html = sprintf( 
      '<%1$s %2$s>%3$s</%1$s>', 
      "span", 
      $this->get_render_attribute_string( "subtitle" ), 
      $subtitle
    );

    $text_html = sprintf( 
      '<%1$s %2$s>%3$s</%1$s>', 
      "p", 
      $this->get_render_attribute_string( "text" ), 
      $text
    );

    $link_html = sprintf( 
      '<%1$s %2$s>%3$s</%1$s>', 
      "a", 
      $this->get_render_attribute_string( "link" ), 
      $link_text
    );

    $content_format = '<div ' . $this->get_render_attribute_string( "content_container" ) . '>%1$s</div>';

    if ( $type == "general-style" ) 
    {
      $content_html = sprintf(
        $content_format, 
        $subtitle_html . $title_html . $text_html . $link_html
      );

      $banner = $image_html . $content_html;
    } 
    else if ( $type == "hover-style" ) 
    {
      $overlay_html = "<span class='overlay'></span>";

      $content_html = sprintf(
        $content_format,
        $title_html . $text_html . $link_html
      );

      $banner = $image_html . $subtitle_html . $overlay_html . $content_html;
    }
    else if ( $type == "creative-style" ) 
    {
      $thumb_html = "
        <div class='img-container target-img-anim sec-img'>
          $thumb
        </div>
      ";

      $content_html = sprintf(
        $content_format,
        $subtitle_html . $title_html . $text_html . $link_html
      );

      $banner = $image_html . $thumb_html . $content_html;
    }

		// Final HTML to render
    $html = sprintf(
      '<%1$s %2$s>%3$s</%1$s>', 
      "div",
      $this->get_render_attribute_string( "banner" ),
      $banner
    );

    echo "$html";
  }

}