<?php

namespace Aventa;

use Elementor\Utils;
use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Repeater;


if ( !defined( "ABSPATH" ) ) exit; 

class Rad_Slideshow_Widget extends Widget_Base{

  public function get_name() {
    return "rad-slideshow";
  }

  public function get_title() {
    return esc_html__("Slideshow", "aventa");
  }

  public function get_keywords() {
		return [ "aventa" ];
	}

  public function get_icon() {
    return "eicon-slider-vertical";
  }

  public function get_categories() {
    return [ Rad_Helpers::WIDGETS_CATEGORY ];
  }

  protected function register_controls() {

		// Widget content
    $this->start_controls_section(
			"section_title",
			[
				"label" => esc_html__( "Slideshow", "aventa" ),
			]
		);

    $this->add_control(
			"ss_main_title", [
				"label" => esc_html__( "Slideshow Title", "aventa" ),
				"type" => Controls_Manager::TEXT,
        "separator" => "after"
			]
		);

    $slideshow_item = new Repeater();

    $slideshow_item->add_control(
			"ss_image",
			[
				"label" => esc_html__( "Image", "aventa" ),
				"type" => Controls_Manager::MEDIA,
				"default" => [
					"url" => Utils::get_placeholder_image_src(),
				]
			]
		); 

		$slideshow_item->add_control(
			"ss_title", [
				"label" => esc_html__( "Title", "aventa" ),
				"type" => Controls_Manager::TEXT,
			]
		);

    $slideshow_item->add_control(
      "ss_has_link",
      [
        "label" => esc_html__( "Has Link?", "aventa" ),
        "type" => Controls_Manager::SWITCHER,
        "label_on" => esc_html__( "Yes", "aventa" ),
        "label_off" => esc_html__( "No", "aventa" ),
        "return_value" => "true"
      ]
    );

    Rad_Helpers::link( $slideshow_item, "ss_link", "Link Text", "Link URL", [ "ss_has_link" => "true" ] );

    $this->add_control(
			"ss_items",
			[
				"label" => esc_html__( "Slideshow Items", "aventa" ),
				"type" => Controls_Manager::REPEATER,
				"fields" => $slideshow_item->get_controls(),
				"title_field" => "{{{ ss_title }}}"
			]
		);

		$this->end_controls_section();

    // General style
    $this->start_controls_section(
			"section_general_style",
			[
				"label" => esc_html__( "General", "aventa" ),
				"tab" => Controls_Manager::TAB_STYLE,
			]
		);

    $this->add_control(
      "ss_background_color",
      [
        "label" => esc_html__( "Background Color", "aventa" ),
				"type" => Controls_Manager::COLOR,
				"selectors" => [
					"{{WRAPPER}} .slideshow-block .sl-items" => "background-color: {{VALUE}};",
				],
        "default" => "#EEE",
        "separator" => "after"
      ]
    );

    Rad_Helpers::typography( 
      $this, 
      "ss_main_title", 
      ".slideshow-block .sl-items > h3" 
    );

    $this->add_control(
      "ss_image_size",
      [
        "label" => esc_html__( "Image Size", "aventa" ),
        "type" => Controls_Manager::SELECT,
        "options" => [
          "full"   => esc_html__( "Full", "aventa" ),
          "medium"   => esc_html__( "Medium", "aventa" ),
          "thumbnail"   => esc_html__( "Thumbnail", "aventa" ),
          "custom"   => esc_html__( "Custom", "aventa" ),
        ],
        "separator" => "before",
        "default" => "full"
      ]
    );

    $this->add_control(
			"ss_image_dimension",
			[
				"label" => esc_html__( "Image Dimension", "aventa" ),
				"type" => Controls_Manager::IMAGE_DIMENSIONS,
				"description" => esc_html__( "Crop the original image size to any custom size.", "aventa" ),
        "condition" => [
          "ss_image_size" => "custom"
        ]
			]
		);

		$this->end_controls_section();

    $this->start_controls_section(
			"section_slideshow_item_style",
			[
				"label" => esc_html__( "Slideshow Items", "aventa" ),
				"tab" => Controls_Manager::TAB_STYLE,
			]
		);

    Rad_Helpers::typography( 
      $this, 
      "ss_title", 
      ".slideshow-block .sl-items ul li", 
      [ "min" => 10, "max" => 40, "step" => 1 ],
      15
    );

		$this->end_controls_section();


  }

  protected function render() {
    $settings = $this->get_settings_for_display();

		if ( !$settings[ "ss_items" ] ) {
			return;
		}

    $classes = [ 
      "slideshow-block",
      "resp-block"
    ];

    // User inputs
		$slideshow_title  = esc_html( $settings[ "ss_main_title" ] );
		$slideshow_items  = $settings[ "ss_items" ];
		$image_size       = esc_attr( $settings[ "ss_image_size" ] );

    if ( $image_size == "custom" ) {
      $image_size = [
        $settings["ss_image_dimension"]["width"], 
        $settings["ss_image_dimension"]["height"]
      ];
    }

    // Attributes
    $this->add_render_attribute( "slideshow_wrapper", "class", implode( " ", $classes ) );

    if ( $slideshow_items ) {

      $item_titles = [];
      $item_images = [];

      foreach ( $slideshow_items as $slideshow_item ) {

        $id         = esc_attr( $slideshow_item[ "_id" ] );
        $title      = esc_html( $slideshow_item[ "ss_title" ] );
        $image_url  = $slideshow_item[ "ss_image" ][ "id" ] 
          ?  wp_get_attachment_image_url( 
              esc_attr( $slideshow_item[ "ss_image" ][ "id" ] ),
              $image_size
            )
          : Utils::get_placeholder_image_src();

        if ( !$title ) continue;
        
        $item_titles[] = "<li data-id='$id' class='underline-style'>$title</li>";
        $item_images[] = "
          <div id='$id' class='slideshow-item'>
            <div
              class='img lazy-load bg-lazy-load lazy-load-on-load' 
              data-src='$image_url'>
            </div>
          </div>";
      }

      // HTML elements
      $title_html = sprintf( 
        '<%1$s>%2$s</%1$s>', 
        "h3", 
        $slideshow_title 
      );

      // HTML structure of the element
      $format = '
        <div ' . $this->get_render_attribute_string( "slideshow_wrapper" ) . '>
          <div class="sl-items">
            %1$s
            <ul>%2$s</ul>
          </div>
          <div class="sl-images">%3$s</div>
        </div>
      ';

      // Final HTML to render
      $html = sprintf( 
        $format,
        $title_html,
        implode( " ", $item_titles ),
        implode( " ", $item_images)
      );

      echo "$html";
    }

  }

}