<?php 

namespace Aventa;

use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Core\Kits\Documents\Tabs\Global_Colors;
use Elementor\Core\Kits\Documents\Tabs\Global_Typography;
use Elementor\Plugin as Elementor;
use \Elementor\Repeater;
use \Elementor\Utils;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Controls helper class
*/
class Rad_Helpers {

	/**
	 * Widget category name
	*/
  const WIDGETS_CATEGORY = "aventa-widgets";
  const WC_WIDGETS_CATEGORY = "aventa-shop-widgets";
	
	/**
	 * Elementor directory path 
	*/
	const ELEMENTOR_DIR_PATH = RAD_THEME_INC_URI . "/admin/compatibility/elementor/";

	/**
	 * Text alignment control
	 * @param array[string] $options You can pass left, right, center 
	*/
	public static function alignment( 
		$widget, 
		$id, 
		$condition = [], 
		$alignments = [], 
		$default = "center",
		$is_responsive = true 
		) {

		if ( !$alignments ) {

			$options =	[
				"left" => [
					"title" => esc_html__( "Left", "aventa" ),
					"icon" => "eicon-text-align-left",
				],
				"center" => [
					"title" => esc_html__( "Center", "aventa" ),
					"icon" => "eicon-text-align-center",
				],
				"right" => [
					"title" => esc_html__( "Right", "aventa" ),
					"icon" => "eicon-text-align-right",
				]
			];

		} else {

			if ( in_array( "left", $alignments ) ) {
				$options[ "left" ] = [
					"title" => esc_html__( "Left", "aventa" ),
					"icon" => "eicon-text-align-left",
				];
			}
			
			if ( in_array( "center", $alignments ) ) {
				$options[ "center" ] = [
					"title" => esc_html__( "Center", "aventa" ),
					"icon" => "eicon-text-align-center",
				];
			}
			
			if ( in_array( "right", $alignments ) ) {
				$options["right"] = [
					"title" => esc_html__( "Right", "aventa" ),
					"icon" => "eicon-text-align-right",
				];
			}

		}

		if ( $is_responsive ) {

			$widget->add_responsive_control(
				$id,
				[
					"label" => esc_html__( "Alignment", "aventa" ),
					"type" => Controls_Manager::CHOOSE,
					"options" => $options,
					"default" => $default,
					"condition" => $condition
				]
			);

		} else {

			$widget->add_control(
				$id,
				[
					"label" => esc_html__( "Alignment", "aventa" ),
					"type" => Controls_Manager::CHOOSE,
					"options" => $options,
					"default" => $default,
					"condition" => $condition
				]
			);

		}

  }

	/**
	 * Heading tag control (H1-H6)
	*/
	public static function heading_tag( $widget, $id, $label = "HTML Tag", $default = "h1" ) {

		$widget->add_control(
			$id,
			[
				"label" => $label,
				"type" => Controls_Manager::SELECT,
				"options" => [
					"h1" => "H1",
					"h2" => "H2",
					"h3" => "H3",
					"h4" => "H4",
					"h5" => "H5",
					"h6" => "H6",
				],
				"default" => $default,
			]
		);

  }

	/**
	 * Grouped typography controls ( Color, Typography, Size )
	*/
	public static function typography( 
		$widget, 
		$id, 
		$selector, 
		$font_size_range = [ "min" => 20, "max" => 80, "step" => 1 ], 
		$default_font_size = 28
	) {

		$widget->add_control(
			$id . "_color",
			[
				"label" => esc_html__( "Text Color", "aventa" ),
				"type" => Controls_Manager::COLOR,
				"global" => [
					"default" => Global_Colors::COLOR_TEXT,
				],
				"selectors" => [
					"{{WRAPPER}} $selector" => "color: {{VALUE}};",
				],
			]
		);

		$widget->add_group_control(
			Group_Control_Typography::get_type(),
      [
        "name" => $id . "_typography",
        "label" => esc_html__( "Typography", "aventa" ),
        "exclude" => [
          "line_height",
          "text_decoration"
        ],
        "global" => [
					"default" => Global_Typography::TYPOGRAPHY_TEXT,
				],
        "selector" => "{{WRAPPER}} $selector",
      ]
		);

    $widget->update_responsive_control( 
      $id . "_typography_font_size", 
      [
        "size_units" => [ "px" ],
        "range" => [
					"px" => $font_size_range,
				],
				"default" => [
					"unit" => "px",
					"size" => $default_font_size,
				],
      ] 
    );
		
		$widget->update_responsive_control( 
      $id . "_typography_letter_spacing", 
      [
        "size_units" => [ "px" ],
        "range" => [
					"px" => [
						"min" => 0,
						"max" => 5,
						"step" => 1,
					],
				],
				"default" => [
					"unit" => "px",
					"size" => 0,
				],
      ] 
    );

	}

	/**
	 * Customized slider control with default values and units
	*/
	public static function slider( 
		$widget, 
		$id,
		$label,
		$selectors = [],
		$default_size = 28,
		$range = [ "min" => 20, "max" => 80, "step" => 1 ],
		$condition = [], 
		$is_responsive = true
	) {

		$options = [
			"label" => $label,
			"type" => Controls_Manager::SLIDER,
			"size_units" => [ "px" ],
			"range" => [
				"px" => $range,
			],
			"default" => [
				"unit" => "px",
				"size" => $default_size,
			],
			"selectors" => $selectors,
			"condition" => $condition,
		];

		if ( $is_responsive ) {
			$widget->add_responsive_control( $id, $options );
		} else {
			$widget->add_control( $id, $options );
		}

	}

	/**
	 * Customized URL control with a title
	*/
	public static function link( $widget, $id, $title_label, $url_label, $condition = [] ) {

		$widget->add_control(
			$id . "_title", 
      [
				"label" => $title_label,
				"type" => Controls_Manager::TEXT,
				"default" => esc_html__( "Link" , "aventa" ),
				"label_block" => true,
				"condition" => $condition
			]
		);

		$widget->add_control(
			$id . "_url",
			[
				"label" => $url_label,
				"type" => Controls_Manager::URL,
				"placeholder" => __( "https://example.com", "aventa" ),
				"show_external" => true,
				"default" => [
					"url" => "",
					"is_external" => true,
					"nofollow" => true,
				],
				"condition" => $condition
			]
		);

	}


	/**
	 * Social accounts input generator
	*/
	public static function social_control( $widget, $id ) {

		$socials_list = [
			"instagram" 	=> esc_html__( "Instagram", "aventa" ) ,
			"facebook" 		=> esc_html__( "Facebook", "aventa" ) ,
			"twitter" 		=> esc_html__( "Twitter", "aventa" ) ,
			"vimeo" 			=> esc_html__( "Vimeo", "aventa" ) ,
			"youtube" 		=> esc_html__( "YouTube", "aventa" ) ,
			"dribbble" 		=> esc_html__( "Dribbble", "aventa" ) ,
			"tumblr" 			=> esc_html__( "Tumbler", "aventa" ) ,
			"linkedin" 		=> esc_html__( "LinkedIn", "aventa" ) ,
			"flickr" 			=> esc_html__( "Flickr", "aventa" ) ,
			"github" 			=> esc_html__( "GitHub", "aventa" ) ,
			"lastfm" 			=> esc_html__( "Last.fm", "aventa" ) ,
			"paypal" 			=> esc_html__( "PayPal", "aventa" ) ,
			"feed" 				=> esc_html__( "RSS", "aventa" ) ,
			"skype" 			=> esc_html__( "Skype", "aventa" ) ,
			"wordpress" 	=> esc_html__( "WordPress", "aventa" ) ,
			"yahoo" 			=> esc_html__( "Yahoo", "aventa" ) ,
			"steam" 			=> esc_html__( "Steam", "aventa" ) ,
			"reddit" 			=> esc_html__( "Reddit", "aventa" ) ,
			"stumbleupon" => esc_html__( "StumbleUpon", "aventa" ) ,
			"pinterest" 	=> esc_html__( "Pinterest", "aventa" ) ,
			"deviantart" 	=> esc_html__( "DeviantArt", "aventa" ) ,
			"xing" 				=> esc_html__( "Xing", "aventa" ) ,
			"blogger" 		=> esc_html__( "Blogger", "aventa" ) ,
			"soundcloud" 	=> esc_html__( "SoundCloud", "aventa" ) ,
			"delicious" 	=> esc_html__( "Delicious", "aventa" ) ,
			"foursquare" 	=> esc_html__( "Foursquare", "aventa" ) ,
			"behance" 		=> esc_html__( "Behance", "aventa" ) 
		];
		
		$social = new Repeater();

		// Is custom social or not
    $social->add_control(
      $id . "_is_custom",
      [
        "label" => esc_html__( "Is Custom?", "aventa" ),
        "type" => Controls_Manager::SWITCHER,
        "label_on" => esc_html__( "Yes", "aventa" ),
        "label_off" => esc_html__( "No", "aventa" ),
        "return_value" => "true",
				"default" => ""
      ]
    );

		// Select social name from available social accounts
    $social->add_control(
      $id . "_name", 
      [
        "label" => esc_html__( "Social Accounts", "aventa" ),
        "type" => Controls_Manager::SELECT2,
				"options" => $socials_list,
				"default" => "instagram",
        "condition" => [
          $id . "_is_custom" => ""
        ]
      ]
    );

		// Custom social account name
		$social->add_control(
			$id . "_custom_name",
			[
				"label" => esc_html__( "Name", "aventa" ),
				"type" => Controls_Manager::TEXT,
				"condition" => [
          $id . "_is_custom" => "true"
        ]
			]
		);

		$social->add_control(
			$id . "_url",
			[
				"label" => esc_html__( "URL", "aventa" ),
				"type" => Controls_Manager::URL,
				"placeholder" => __( "https://example.com", "aventa" ),
				"show_external" => true,
				"default" => [
					"url" => "",
					"is_external" => true,
					"nofollow" => true,
				],
			]
		);

		// Social icon
    $social->add_control(
			$id . "_custom_icon",
			[
				"label" => esc_html__( "Icon", "aventa" ),
				"type" => Controls_Manager::MEDIA,
				"default" => [
					"url" => Utils::get_placeholder_image_src(),
				],
				"condition" => [
          $id . "_is_custom" => "true"
        ]
			]
		);

		$widget->add_control(
			$id,
			[
				"label" => esc_html__( "Social Accounts", "aventa" ),
				"type" => Controls_Manager::REPEATER,
				"prevent_empty" => false,
				"fields" => $social->get_controls(),
				"title_field" =>  "Social Item"
			]
		);

	}

	/**
	 * Show placeholder image
	*/
	public static function get_placeholder_image() {
		return "<img src='" . Utils::get_placeholder_image_src() . "' />";
	}

	/**
	 * Check whether it's elementor editor mode
	*/
	public static function is_edit_mode() {
		return Elementor::$instance->editor->is_edit_mode();
	}

	/**
	 * Returns an array of all WooCommerce products 
	*/
  public static function get_products() {
		if ( function_exists( "wc_get_products" ) ) {
			$args = [
				'limit'  => -1,
				'status' => 'publish'
			];

			$products = wc_get_products( $args );
			
			if ( $products ) {
				$products_array = [];
				foreach ($products as $product) {
					$products_array[ $product->get_id() ] = $product->get_id() . " - " . $product->get_title();
				}
				return $products_array;
			}
		}
  }

	/**
	 * Returns an array of all WooCommerce categories
	*/
  public static function get_wc_categories() {
		$args = [
			"taxonomy"   => "product_cat"
		];

		$terms = get_terms( $args );

    if ( $terms ) {
      $terms_array = [];
      foreach ( $terms as $term ) {
        $terms_array[ $term->term_id ] = $term->name;
      }
      return $terms_array;
    }
  }

	/**
	 * Returns an array of all WooCommerce tags
	*/
  public static function get_wc_tags() {
		$args = [
			"taxonomy"   => "product_tag"
		];

		$terms = get_terms( $args );

    if ( $terms ) {
      $terms_array = [];
      foreach ( $terms as $term ) {
        $terms_array[ $term->slug ] = $term->name;
      }
      return $terms_array;
    }
  }
 
}

?>